# SPDX-License-Identifier: Apache-2.0
# Copyright 2023 The Meson development team
from __future__ import annotations

from pathlib import Path

from .generatorbase import GeneratorBase
from .model import ReferenceManual


class GeneratorVim(GeneratorBase):
    def __init__(self, manual: ReferenceManual, out_dir: Path) -> None:
        super().__init__(manual)
        self.out_dir = out_dir

    def generate(self) -> None:
        template_dir = Path(__file__).resolve().parent / 'templates'
        outname = 'meson.vim'
        template_name = f'{outname}.mustache'
        template_file = template_dir / template_name

        builtin_funcs = [f.name for f in self.sorted_and_filtered(self.functions)]
        data = {
            'builtin_funcs': '\n  \\ '.join(builtin_funcs)
        }

        # Import here, so that other generators don't also depend on it
        import chevron
        result = chevron.render(
            template=template_file.read_text(encoding='utf-8'),
            data=data,
            warn=True,
        )

        self.out_dir.mkdir(parents=True, exist_ok=True)
        out_file = self.out_dir / outname
        out_file.write_text(result, encoding='utf-8')
