/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 * Copyright by The HDF Group.                                               *
 * Copyright by the Board of Trustees of the University of Illinois.         *
 * All rights reserved.                                                      *
 *                                                                           *
 * This file is part of HDF5.  The full HDF5 copyright notice, including     *
 * terms governing use, modification, and redistribution, is contained in    *
 * the COPYING file, which can be found at the root of the source code       *
 * distribution tree, or in https://www.hdfgroup.org/licenses.               *
 * If you do not have access to either file, you may request a copy from     *
 * help@hdfgroup.org.                                                        *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

#include <string>

#include "H5private.h" // for HDmemset
#include "H5Include.h"
#include "H5Exception.h"
#include "H5IdComponent.h"
#include "H5PropList.h"
#include "H5FaccProp.h"
#include "H5FcreatProp.h"
#include "H5OcreatProp.h"
#include "H5DcreatProp.h"
#include "H5StrcreatProp.h"
#include "H5LcreatProp.h"
#include "H5LaccProp.h"
#include "H5Location.h"
#include "H5Object.h"
#include "H5DataType.h"
#include "H5DataSpace.h"
#include "H5AbstractDs.h"
#include "H5CommonFG.h"
#include "H5Group.h"
#include "H5File.h"

namespace H5 {

#ifndef DOXYGEN_SHOULD_SKIP_THIS
// userAttrOpWrpr simply interfaces between the user's function and the
// C library function H5Aiterate2; used to resolve the different prototype
// problem.  May be moved to Iterator later.
extern "C" herr_t
userAttrOpWrpr(hid_t loc_id, const char *attr_name, const H5A_info_t *ainfo, void *op_data)
{
    H5std_string s_attr_name = H5std_string(attr_name);
#ifdef NO_STATIC_CAST
    UserData4Aiterate *myData = (UserData4Aiterate *)op_data;
#else
    UserData4Aiterate *myData = static_cast<UserData4Aiterate *>(op_data);
#endif
    myData->op(*myData->location, s_attr_name, myData->opData);
    return 0;
}

//--------------------------------------------------------------------------
// Function:    H5Location default constructor (protected)
// Programmer   Binh-Minh Ribler - 2000
//--------------------------------------------------------------------------
H5Location::H5Location() : IdComponent()
{
}

//--------------------------------------------------------------------------
// Function:    H5Location overloaded constructor (protected)
// Purpose      Creates an H5Location object using the id of an existing HDF5
//              object.
// Parameters   object_id - IN: Id of an existing HDF5 object
// Programmer   Binh-Minh Ribler - 2000

// *** Deprecation warning ***
// This constructor is no longer appropriate because the data member "id" had
// been moved to the sub-classes.  It will be removed in 1.10 release.  If its
// removal does not raise any problems in 1.10, it will be removed from 1.8 in
// subsequent releases.
// Removed in 1.8.18 and 1.10.1 - Aug 2016
//--------------------------------------------------------------------------
// H5Location::H5Location(const hid_t object_id) : IdComponent() {}

//--------------------------------------------------------------------------
// Function:    H5Location copy constructor
// Purpose:     This noop copy constructor is removed as a result of the data
//              member "id" being moved down to sub-classes. (Mar 2015)
///\param       original - IN: H5Location instance to copy
// Programmer   Binh-Minh Ribler - 2000
//
// *** Deprecation warning ***
// This constructor is no longer appropriate because the data member "id" had
// been moved to the sub-classes.  It is removed from 1.8.15 because it is
// a noop and it can be generated by the compiler if needed.
//--------------------------------------------------------------------------
// H5Location::H5Location(const H5Location& original) : IdComponent() {}

#endif // DOXYGEN_SHOULD_SKIP_THIS

/***************************************************************************
                           Notes for H5A wrappers
                           ======================
May, 2017 (1.8.19)
        The H5A wrappers are copied to H5Object to prevent an attribute
        id being passed in to H5A APIs.  The original H5A wrapper functions
        here will be deprecated in future releases.
May, 2018 (1.8.21)
        They are removed from 1.8.21

***************************************************************************/

//--------------------------------------------------------------------------
// Function:    H5Location::iterateAttrs
///\brief       Iterates a user's function over all the attributes of an H5
///             object, which may be a group, dataset or named datatype.
///\param       user_op - IN: User's function to operate on each attribute
///\param       _idx - IN/OUT: Starting (IN) and ending (OUT) attribute indices
///\param       op_data - IN: User's data to pass to user's operator function
///\return      Returned value of the last operator if it was non-zero, or
///             zero if all attributes were processed
///\exception   H5::AttributeIException
///\par Description
///             The signature of user_op is
///             void (*)(H5::H5Location&, H5std_string, void*).
///             For information, please refer to the C layer Reference Manual
///             at:
/// https://support.hdfgroup.org/HDF5/doc/RM/RM_H5A.html#Annot-Iterate
//--------------------------------------------------------------------------
int
H5Location::iterateAttrs(attr_operator_t user_op, unsigned *_idx, void *op_data)
{
    // store the user's function and data
    UserData4Aiterate *userData = new UserData4Aiterate;
    userData->opData            = op_data;
    userData->op                = user_op;
    userData->location          = this;

    // call the C library routine H5Aiterate2 to iterate the attributes
    hsize_t idx = _idx ? static_cast<hsize_t>(*_idx) : 0;
    int     ret_value =
        H5Aiterate2(getId(), H5_INDEX_NAME, H5_ITER_INC, &idx, userAttrOpWrpr, static_cast<void *>(userData));

    // release memory
    delete userData;

    if (ret_value >= 0) {
        /* Pass back update index value to calling code */
        if (_idx)
            *_idx = static_cast<unsigned>(idx);

        return (ret_value);
    }
    else // raise exception when H5Aiterate returns a negative value
        throw AttributeIException(inMemFunc("iterateAttrs"), "H5Aiterate2 failed");
}

//--------------------------------------------------------------------------
// Function:    H5Location::getNumAttrs
///\brief       Returns the number of attributes attached to an HDF5 object.
///\return      Number of attributes
///\exception   H5::AttributeIException
// Modification
//          - Moved from H5Object to allow passing an attribute id to the
//            C API, in 1.8.20.
//--------------------------------------------------------------------------
int
H5Location::getNumAttrs() const
{
    H5O_info_t objinfo; /* Object info */

    if (H5Oget_info(getId(), &objinfo) < 0)
        throwException("getNumAttrs", "H5Oget_info failed");
    return (static_cast<int>(objinfo.num_attrs));
}

//--------------------------------------------------------------------------
// Function:    H5Location::nameExists
///\brief       Checks if a link of a given name exists in a location.
///\param       name - IN: Searched name - \c char*
///\param       lapl - IN: Link access property list
///\exception   One of the following:
///                 H5::GroupIException
///                 H5::FileIException
///                 H5::DataSetIException
///                 H5::DataTypeIException
///                 H5::AttributeIException
// December 2016
//--------------------------------------------------------------------------
bool
H5Location::nameExists(const char *name, const LinkAccPropList &lapl) const
{
    htri_t ret_value = H5Lexists(getId(), name, lapl.getId());
    if (ret_value > 0)
        return true;
    else if (ret_value == 0)
        return false;
    else // Raise exception when H5Lexists returns a negative value
    {
        throwException("nameExists", "H5Lexists failed");
    }
}

//--------------------------------------------------------------------------
// Function:    H5Location::nameExists
///\brief       Checks if a link of a given name exists in a location.
///\param       name - IN: Searched name - \c H5std_string
///\param       lapl - IN: Link access property list
///\exception   One of the following:
///                 H5::GroupIException
///                 H5::FileIException
///                 H5::DataSetIException
///                 H5::DataTypeIException
///                 H5::AttributeIException
// December 2016
//--------------------------------------------------------------------------
bool
H5Location::nameExists(const H5std_string &name, const LinkAccPropList &lapl) const
{
    return (nameExists(name.c_str(), lapl));
}

//--------------------------------------------------------------------------
// Function:    H5Location::flush
///\brief       Flushes all buffers associated with a location to disk.
///\param       scope - IN: Specifies the scope of the flushing action,
///             which can be either of these values:
///             \li \c H5F_SCOPE_GLOBAL - Flushes the entire virtual file
///             \li \c H5F_SCOPE_LOCAL - Flushes only the specified file
///\exception   One of the following:
///                 H5::GroupIException
///                 H5::FileIException
///                 H5::DataSetIException
///                 H5::DataTypeIException
///                 H5::AttributeIException
///\par Description
///             This location is used to identify the file to be flushed.
// December 2005
// Modification
//      Sep 2012 - BMR
//              Moved from H5File/H5Object
//--------------------------------------------------------------------------
void
H5Location::flush(H5F_scope_t scope) const
{
    herr_t ret_value = H5Fflush(getId(), scope);
    if (ret_value < 0) {
        throwException("flush", "H5Fflush failed");
    }
}

//--------------------------------------------------------------------------
// Function:    H5Location::getFileName
///\brief       Gets the name of the file, in which this HDF5 object belongs.
///\return      File name
///\exception   One of the following:
///                 H5::GroupIException
///                 H5::FileIException
///                 H5::DataSetIException
///                 H5::DataTypeIException
///                 H5::AttributeIException
// July 2004
//--------------------------------------------------------------------------
H5std_string
H5Location::getFileName() const
{
    try {
        return (p_get_file_name());
    }
    catch (IdComponentException &E) {
        throwException("getFileName", E.getDetailMsg());
    }
}

//--------------------------------------------------------------------------
// Function:    H5Location::getObjectInfo
///\brief       Retrieve information about an object, specified by location.
///
///\exception   One of the following:
///                 H5::GroupIException
///                 H5::FileIException
///                 H5::DataSetIException
///                 H5::DataTypeIException
///                 H5::AttributeIException
// August 2017
//--------------------------------------------------------------------------
void
H5Location::getObjectInfo(H5O_info_t *objinfo) const
{
    herr_t ret_value = H5Oget_info(getId(), objinfo);
    if (ret_value < 0) {
        throwException("getObjectInfo", "H5Oget_info failed");
    }
}

//--------------------------------------------------------------------------
// Function:    H5Location::getObjectInfo
///\brief       Retrieve information about an object, specified by name.
///
///\exception   One of the following:
///                 H5::GroupIException
///                 H5::FileIException
///                 H5::DataSetIException
///                 H5::DataTypeIException
///                 H5::AttributeIException
// August 2017
//--------------------------------------------------------------------------
void
H5Location::getObjectInfo(const char *name, H5O_info_t *objinfo, const LinkAccPropList &lapl) const
{
    herr_t ret_value = H5Oget_info_by_name(getId(), name, objinfo, lapl.getId());
    if (ret_value < 0) {
        throwException("getObjectInfo", "H5Oget_info_by_name failed");
    }
}

//--------------------------------------------------------------------------
// Function:    H5Location::getObjectInfo
///\brief       Retrieve information about an object, specified by name.
///
///\exception   One of the following:
///                 H5::GroupIException
///                 H5::FileIException
///                 H5::DataSetIException
///                 H5::DataTypeIException
///                 H5::AttributeIException
// August 2017
//--------------------------------------------------------------------------
void
H5Location::getObjectInfo(const H5std_string &name, H5O_info_t *objinfo, const LinkAccPropList &lapl) const
{
    herr_t ret_value = H5Oget_info_by_name(getId(), name.c_str(), objinfo, lapl.getId());
    if (ret_value < 0) {
        throwException("getObjectInfo", "H5Oget_info_by_name failed");
    }
}

//--------------------------------------------------------------------------
// Function:    H5Location::objVersion
///\brief       Returns the header version of this HDF5 object.
///\return      Object version, which can have the following values:
///             \li \c H5O_VERSION_1
///             \li \c H5O_VERSION_2
///\exception   One of the following:
///                 H5::GroupIException
///                 H5::FileIException
///                 H5::DataSetIException
///                 H5::DataTypeIException
///                 H5::AttributeIException
///             Exception will be thrown when:
///             - an error returned by the C API
///             - version number is not one of the valid values above
// August 2017
//--------------------------------------------------------------------------
unsigned
H5Location::objVersion() const
{
    H5O_info_t objinfo;
    unsigned   version = 0;

    // Use C API to get information of the object
    herr_t ret_value = H5Oget_info(getId(), &objinfo);

    // Throw exception if C API returns failure
    if (ret_value < 0)
        throwException("objVersion", "H5Oget_info failed");
    // Return a valid version or throw an exception for invalid value
    else {
        version = objinfo.hdr.version;
        if (version != H5O_VERSION_1 && version != H5O_VERSION_2)
            throwException("objVersion", "Invalid version for object");
    }
    return (version);
}

//--------------------------------------------------------------------------
// Function:    H5Location::setComment
///\brief       Sets or resets the comment for an object specified by its name.
///\param       name  - IN: Name of the object
///\param       comment - IN: New comment
///\exception   One of the following:
///                 H5::GroupIException
///                 H5::FileIException
///                 H5::DataSetIException
///                 H5::DataTypeIException
///                 H5::AttributeIException
///\par Description
///             If \a comment is an empty string or a null pointer, the comment
///             message is removed from the object.
///             Comments should be relatively short, null-terminated, ASCII
///             strings.  They can be attached to any object that has an
///             object header, e.g., data sets, groups, named data types,
///             and data spaces, but not symbolic links.
// July 2004
// Modification
//      Moved from CommonFG, Sep 2013
//--------------------------------------------------------------------------
void
H5Location::setComment(const char *name, const char *comment) const
{
    herr_t ret_value = H5Oset_comment_by_name(getId(), name, comment, H5P_DEFAULT);
    if (ret_value < 0)
        throwException("setComment", "H5Oset_comment_by_name failed");
}

//--------------------------------------------------------------------------
// Function:    H5Location::setComment
///\brief       This is an overloaded member function, provided for convenience.
///             It differs from the above function in that it takes an
///             \c H5std_string for \a name and \a comment.
// July 2004
// Modification
//      Moved from CommonFG, Sep 2013
//--------------------------------------------------------------------------
void
H5Location::setComment(const H5std_string &name, const H5std_string &comment) const
{
    setComment(name.c_str(), comment.c_str());
}

//--------------------------------------------------------------------------
// Function:    H5Location::setComment
///\brief       This is an overloaded member function, provided for convenience.
///             It differs from the above function in that it doesn't take
///             an object name.
///\exception   One of the following:
///                 H5::GroupIException
///                 H5::FileIException
///                 H5::DataSetIException
///                 H5::DataTypeIException
///                 H5::AttributeIException
// Sep 2013
//--------------------------------------------------------------------------
void
H5Location::setComment(const char *comment) const
{
    herr_t ret_value = H5Oset_comment_by_name(getId(), ".", comment, H5P_DEFAULT);
    if (ret_value < 0)
        throwException("setComment", "H5Oset_comment_by_name failed");
}

//--------------------------------------------------------------------------
// Function:    H5Location::setComment
///\brief       This is an overloaded member function, provided for convenience.
///             It differs from the above function in that it takes an
///             \c H5std_string for \a comment.
// Sep 2013
//--------------------------------------------------------------------------
void
H5Location::setComment(const H5std_string &comment) const
{
    setComment(comment.c_str());
}

//--------------------------------------------------------------------------
// Function:    H5Location::removeComment
///\brief       Removes the comment from an object specified by its name.
///\param       name  - IN: Name of the object
///\exception   One of the following:
///                 H5::GroupIException
///                 H5::FileIException
///                 H5::DataSetIException
///                 H5::DataTypeIException
///                 H5::AttributeIException
// July 2004
// Modification
//      Moved from CommonFG, Sep 2013
//--------------------------------------------------------------------------
void
H5Location::removeComment(const char *name) const
{
    herr_t ret_value = H5Oset_comment_by_name(getId(), name, NULL, H5P_DEFAULT);
    if (ret_value < 0)
        throwException("removeComment", "H5Oset_comment_by_name failed");
}

//--------------------------------------------------------------------------
// Function:    H5Location::removeComment
///\brief       This is an overloaded member function, provided for convenience.
///             It differs from the above function in that it takes an
///             \c H5std_string for \a name.
// July 2004
// Modification
//      Moved from CommonFG, Sep 2013
//--------------------------------------------------------------------------
void
H5Location::removeComment(const H5std_string &name) const
{
    removeComment(name.c_str());
}

//--------------------------------------------------------------------------
// Function:    H5Location::getComment
///\brief       Retrieves the comment for this location, returning its length.
///\param       name     - IN: Name of the object
///\param       buf_size - IN: Length of the comment to retrieve
///\param       comment  - OUT: Retrieved comment
///\return      Actual length of the comment
///\exception   One of the following:
///                 H5::GroupIException
///                 H5::FileIException
///                 H5::DataSetIException
///                 H5::DataTypeIException
///                 H5::AttributeIException
///\par Description
///             This function retrieves \a buf_size characters of the comment
///             including the null terminator.  Thus, if the actual length
///             of the comment is more than buf_size-1, the retrieved comment
///             will be truncated to accommodate the null terminator.
// March 2014
//--------------------------------------------------------------------------
ssize_t
H5Location::getComment(const char *name, size_t buf_size, char *comment) const
{
    // H5Oget_comment_by_name will get buf_size chars of the comment including
    // the null terminator
    ssize_t comment_len;
    comment_len = H5Oget_comment_by_name(getId(), name, comment, buf_size, H5P_DEFAULT);

    // If H5Oget_comment_by_name returns a negative value, raise an exception
    if (comment_len < 0) {
        throwException("getComment", "H5Oget_comment_by_name failed");
    }
    // If the comment is longer than the provided buffer size, the C library
    // will not null terminate it
    if (static_cast<size_t>(comment_len) >= buf_size)
        comment[buf_size - 1] = '\0';

    // Return the actual comment length, which might be different from buf_size
    return (comment_len);
}

//--------------------------------------------------------------------------
// Function:    H5Location::getComment
///\brief       Returns the comment as \a string for this location,
///             returning its length.
///\param       name     - IN: Name of the object
///\param       buf_size - IN: Length of the comment to retrieve, default to 0
///\return      Comment string
///\exception   One of the following:
///                 H5::GroupIException
///                 H5::FileIException
///                 H5::DataSetIException
///                 H5::DataTypeIException
///                 H5::AttributeIException
// July 2004
// Modification
//      Moved from CommonFG, Sep 2013
//--------------------------------------------------------------------------
H5std_string
H5Location::getComment(const char *name, size_t buf_size) const
{
    // Initialize string to "", so that if there is no comment, the returned
    // string will be empty
    H5std_string comment("");

    // Preliminary call to get the comment's length
    ssize_t comment_len = H5Oget_comment_by_name(getId(), name, NULL, (size_t)0, H5P_DEFAULT);

    // If H5Oget_comment_by_name returns a negative value, raise an exception
    if (comment_len < 0) {
        throwException("getComment", "H5Oget_comment_by_name failed");
    }

    // If comment exists, calls C routine again to get it
    else if (comment_len > 0) {
        size_t tmp_len = buf_size;

        // If buffer size is not provided, use comment length
        if (tmp_len == 0)
            tmp_len = comment_len;

        // Temporary buffer for char* comment
        char *comment_C = new char[tmp_len + 1];
        HDmemset(comment_C, 0, tmp_len + 1); // clear buffer

        // Used overloaded function
        ssize_t temp_len = getComment(name, tmp_len + 1, comment_C);
        if (temp_len < 0) {
            delete[] comment_C;
            throwException("getComment", "H5Oget_comment_by_name failed");
        }

        // Convert the C comment to return
        comment = comment_C;

        // Clean up resource
        delete[] comment_C;
    }

    // Return the string comment
    return (comment);
}

//--------------------------------------------------------------------------
// Function:    H5Location::getComment
///\brief       This is an overloaded member function, provided for convenience.
///             It differs from the above function in that it takes an
///             \c H5std_string for \a name.
///\exception   One of the following:
///                 H5::GroupIException
///                 H5::FileIException
///                 H5::DataSetIException
///                 H5::DataTypeIException
///                 H5::AttributeIException
// July 2004
// Modification
//      Moved from CommonFG, Sep 2013
//--------------------------------------------------------------------------
H5std_string
H5Location::getComment(const H5std_string &name, size_t buf_size) const
{
    return (getComment(name.c_str(), buf_size));
}

//--------------------------------------------------------------------------
// Function:    H5Location::openObjId
///\brief       Opens an object without knowing the object type.
///\param       obj_name - IN: Path to the object
///\param       lapl     - IN: Access plist for the link pointing to the object
///\exception   One of the following:
///                 H5::GroupIException
///                 H5::FileIException
///                 H5::DataSetIException
///                 H5::DataTypeIException
///                 H5::AttributeIException
///\par Description
///             This function opens an object at this location, using
///             H5Oopen.  Thus, an object can be opened without knowing
///             its type.
// May 2017
//--------------------------------------------------------------------------
hid_t
H5Location::openObjId(const char *obj_name, const LinkAccPropList &lapl) const
{
    hid_t ret_value = H5Oopen(getId(), obj_name, lapl.getId());
    if (ret_value < 0) {
        throwException("openObjId", "H5Oopen failed");
    }
    return (ret_value);
}

//--------------------------------------------------------------------------
// Function:    H5Location::openObjId
///\brief       This is an overloaded member function, provided for convenience.
///             It takes a reference to a \c H5std_string for the object's path.
///\param       obj_name - IN: Path to the object
///\param       lapl     - IN: Access property list for the link pointing to
///                            the object
///\exception   One of the following:
///                 H5::GroupIException
///                 H5::FileIException
///                 H5::DataSetIException
///                 H5::DataTypeIException
///                 H5::AttributeIException
// May 2017
//--------------------------------------------------------------------------
hid_t
H5Location::openObjId(const H5std_string &obj_name, const LinkAccPropList &lapl) const
{
    return (openObjId(obj_name.c_str(), lapl));
}

//--------------------------------------------------------------------------
// Function:    H5Location::closeObjId
///\brief       Closes an object, which was opened with H5Location::openObjId
///
///\exception   H5::LocationException
// May 2017
//--------------------------------------------------------------------------
void
H5Location::closeObjId(hid_t obj_id)
{
    herr_t ret_value = H5Oclose(obj_id);
    if (ret_value < 0) {
        throw LocationException("closeObjId", "H5Oclose failed");
    }
}

#ifndef DOXYGEN_SHOULD_SKIP_THIS

//--------------------------------------------------------------------------
// Function:    H5Location::p_reference (protected)
// Purpose      Creates a reference to an HDF5 object or a dataset region.
// Parameters
//              name - IN: Name of the object to be referenced
//              dataspace - IN: Dataspace with selection
//              ref_type - IN: Type of reference; default to \c H5R_DATASET_REGION
// Exception    H5::ReferenceException
// Programmer   Binh-Minh Ribler - May, 2004
//--------------------------------------------------------------------------
void
H5Location::p_reference(void *ref, const char *name, hid_t space_id, H5R_type_t ref_type) const
{
    herr_t ret_value = H5Rcreate(ref, getId(), name, ref_type, space_id);
    if (ret_value < 0) {
        throw ReferenceException(inMemFunc("reference"), "H5Rcreate failed");
    }
}

#endif // DOXYGEN_SHOULD_SKIP_THIS

//--------------------------------------------------------------------------
// Function:    H5Location::reference
///\brief       Creates a reference to an HDF5 object or a dataset region.
///\param       ref - IN: Reference pointer
///\param       name - IN: Name of the object to be referenced
///\param       dataspace - IN: Dataspace with selection
///\param       ref_type - IN: Type of reference to query, valid values are:
///             \li \c H5R_OBJECT         - Reference is an object reference.
///             \li \c H5R_DATASET_REGION - Reference is a dataset region
///                     reference. (default)
///\exception   H5::ReferenceException
///\note        This method is more suitable for a dataset region reference.
// Programmer   Binh-Minh Ribler - May, 2004
//--------------------------------------------------------------------------
void
H5Location::reference(void *ref, const char *name, const DataSpace &dataspace, H5R_type_t ref_type) const
{
    try {
        p_reference(ref, name, dataspace.getId(), ref_type);
    }
    catch (ReferenceException &E) {
        throw ReferenceException(inMemFunc("reference"), E.getDetailMsg());
    }
}

//--------------------------------------------------------------------------
// Function:    H5Location::reference
///\brief       This is an overloaded member function, provided for convenience.
///             It differs from the above function in that it takes an
///             \c H5std_string for \a name.
///\param       ref - IN: Reference pointer
///\param       name - IN: Name of the object to be referenced
///\param       dataspace - IN: Dataspace with selection
///\param       ref_type - IN: Type of reference to query, valid values are:
///             \li \c H5R_OBJECT         - Reference is an object reference.
///             \li \c H5R_DATASET_REGION - Reference is a dataset region
///                     reference. (default)
///\exception   H5::ReferenceException
///\note        This method is more suitable for a dataset region reference.
// Programmer   Binh-Minh Ribler - May, 2004
//--------------------------------------------------------------------------
void
H5Location::reference(void *ref, const H5std_string &name, const DataSpace &dataspace,
                      H5R_type_t ref_type) const
{
    try {
        p_reference(ref, name.c_str(), dataspace.getId(), ref_type);
    }
    catch (ReferenceException &E) {
        throw ReferenceException(inMemFunc("reference"), E.getDetailMsg());
    }
}

//--------------------------------------------------------------------------
// Function:    H5Location::reference
///\brief       This is an overloaded function, provided for your convenience.
///             It differs from the above function in that it does not take
///             a DataSpace object and the reference type must be specified.
///\param       ref - IN: Reference pointer
///\param       name - IN: Name of the object to be referenced
///\param       ref_type - IN: Type of reference to query, valid values are:
///             \li \c H5R_OBJECT         - Reference is an object reference (default)
///             \li \c H5R_DATASET_REGION - Reference is a dataset region
///\exception   H5::ReferenceException
///\note        This method is more suitable for an object reference.
// Programmer   Binh-Minh Ribler - May, 2004
//--------------------------------------------------------------------------
void
H5Location::reference(void *ref, const char *name, H5R_type_t ref_type) const
{
    try {
        p_reference(ref, name, -1, ref_type);
    }
    catch (ReferenceException &E) {
        throw ReferenceException(inMemFunc("reference"), E.getDetailMsg());
    }
}

//--------------------------------------------------------------------------
// Function:    H5Location::reference
///\brief       This is an overloaded function, provided for your convenience.
///             It differs from the above function in that it takes an
///             \c H5std_string for the object's name.
///\param       ref - IN: Reference pointer
///\param       name - IN: Name of the object to be referenced - \c H5std_string
///\param       ref_type - IN: Type of reference to query, valid values are:
///             \li \c H5R_OBJECT         - Reference is an object reference (default)
///             \li \c H5R_DATASET_REGION - Reference is a dataset region
///\note        This method is more suitable for an object reference.
// Programmer   Binh-Minh Ribler - May, 2004
//--------------------------------------------------------------------------
void
H5Location::reference(void *ref, const H5std_string &name, H5R_type_t ref_type) const
{
    reference(ref, name.c_str(), ref_type);
}

#ifndef DOXYGEN_SHOULD_SKIP_THIS
//--------------------------------------------------------------------------
// Function:    H5Location::p_dereference (protected)
// Purpose      Dereference a ref into an hdf5 object.
// Parameters
//              loc_id - IN: An hdf5 identifier specifying the location of the
//                       referenced object
//              ref - IN: Reference pointer
//              ref_type - IN: Reference type
// Exception    H5::ReferenceException
// Programmer   Binh-Minh Ribler - Oct, 2006
// Modification
//      May 2008 - BMR
//              Moved from IdComponent.
//--------------------------------------------------------------------------
hid_t
H5Location::p_dereference(hid_t loc_id, const void *ref, H5R_type_t ref_type, const char *from_func)
{
    hid_t temp_id = H5Rdereference(loc_id, ref_type, ref);
    if (temp_id < 0) {
        throw ReferenceException(inMemFunc(from_func), "H5Rdereference failed");
    }

    return (temp_id);
}
#endif // DOXYGEN_SHOULD_SKIP_THIS

//--------------------------------------------------------------------------
// Function:    H5Location::dereference
///\brief       Dereferences a reference into an HDF5 object, given an HDF5 object.
///\param       loc - IN: Location of the referenced object
///\param       ref - IN: Reference pointer
///\param       ref_type - IN: Reference type
///\exception   H5::ReferenceException
// Programmer   Binh-Minh Ribler - Oct, 2006
// Modification
//      May, 2008
//              Corrected missing parameters. - BMR
//--------------------------------------------------------------------------
void
H5Location::dereference(const H5Location &loc, const void *ref, H5R_type_t ref_type)
{
    p_setId(p_dereference(loc.getId(), ref, ref_type, "dereference"));
}

//--------------------------------------------------------------------------
// Function:    H5Location::dereference
// brief        Dereferences a reference into an HDF5 object, given an attribute.
// param        attr - IN: Attribute specifying the location of the referenced object
// param        ref - IN: Reference pointer
// param        ref_type - IN: Reference type
// exception    H5::ReferenceException
// October 2006
// Modification
//      May, 2008
//              Corrected missing parameters. - BMR
//      May, 2018
//              Removed in 1.8.21 because H5Location is Attribute's baseclass
//              now. -BMR
//--------------------------------------------------------------------------
// void H5Location::dereference(const Attribute& attr, const void* ref, H5R_type_t ref_type)
//{
// p_setId(p_dereference(attr.getId(), ref, ref_type, "dereference"));
//}

#ifndef H5_NO_DEPRECATED_SYMBOLS
//--------------------------------------------------------------------------
// Function:    H5Location::getObjType
///\brief       Retrieves the type of object that an object reference points to.
///\param       ref_type - IN: Type of reference to query, valid values are:
///             \li \c H5R_OBJECT - Reference is an object reference.
///             \li \c H5R_DATASET_REGION - Reference is a dataset region reference.
///\param       ref      - IN: Reference to query
///\return      An object type, which can be one of the following:
///             \li \c H5G_UNKNOWN  - A failure occurs. (-1)
///             \li \c H5G_GROUP  - Object is a group.
///             \li \c H5G_DATASET - Object is a dataset.
///             \li \c H5G_TYPE Object - is a named datatype
///             \li \c H5G_LINK  - Object is a symbolic link.
///             \li \c H5G_UDLINK  - Object is a user-defined link.
///\exception   H5::ReferenceException
// Programmer   Binh-Minh Ribler - May, 2004
// Modification
//      Sep 2012: Moved up from H5File, Group, DataSet, and DataType
//--------------------------------------------------------------------------
H5G_obj_t
H5Location::getObjType(void *ref, H5R_type_t ref_type) const
{
    try {
        return (p_get_obj_type(ref, ref_type));
    }
    catch (ReferenceException &E) {
        throw ReferenceException(inMemFunc("getObjType"), E.getDetailMsg());
    }
}

#ifndef DOXYGEN_SHOULD_SKIP_THIS
//--------------------------------------------------------------------------
// Function:    H5Location::p_get_obj_type (protected)
// Purpose      Retrieves the type of object that an object reference points to.
// Parameters
//              ref      - IN: Reference to query
//              ref_type - IN: Type of reference to query
// Return       An object type, which can be one of the following:
//                      H5G_UNKNOWN \tFailure occurs (-1)
//                      H5G_GROUP \tObject is a group.
//                      H5G_DATASET \tObject is a dataset.
//                      H5G_TYPE Object \tis a named datatype.
//                      H5G_LINK \tObject is a symbolic link.
//                      H5G_UDLINK \tObject is a user-defined link.
// Exception    H5::ReferenceException
// Programmer   Binh-Minh Ribler - May, 2004
//--------------------------------------------------------------------------
H5G_obj_t
H5Location::p_get_obj_type(void *ref, H5R_type_t ref_type) const
{
    H5G_obj_t obj_type = H5Rget_obj_type1(getId(), ref_type, ref);

    if (obj_type == H5G_UNKNOWN) {
        throw ReferenceException(inMemFunc("getObjType"), "H5Rget_obj_type1 failed");
    }
    return (obj_type);
}
#endif // DOXYGEN_SHOULD_SKIP_THIS
#endif /* H5_NO_DEPRECATED_SYMBOLS */

//--------------------------------------------------------------------------
// Function:    H5Location::getRefObjType
///\brief       Retrieves the type of object that an object reference points to.
///\param       ref      - IN: Reference to query
///\param       ref_type - IN: Type of reference to query, valid values are:
///             \li \c H5R_OBJECT         - Reference is an object reference.
///             \li \c H5R_DATASET_REGION - Reference is a dataset region reference.
///\return      An object type, which can be one of the following:
///             \li \c H5O_TYPE_UNKNOWN - Unknown object type (-1)
///             \li \c H5O_TYPE_GROUP   - Object is a group
///             \li \c H5O_TYPE_DATASET - Object is a dataset
///             \li \c H5O_TYPE_NAMED_DATATYPE - Object is a named datatype
///             \li \c H5O_TYPE_NTYPES  - Number of different object types
///\exception   H5::ReferenceException
// Programmer   Binh-Minh Ribler - May, 2004
//--------------------------------------------------------------------------
H5O_type_t
H5Location::getRefObjType(void *ref, H5R_type_t ref_type) const
{
    try {
        return (p_get_ref_obj_type(ref, ref_type));
    }
    catch (ReferenceException &E) {
        throw ReferenceException(inMemFunc("getRefObjType"), E.getDetailMsg());
    }
}

#ifndef DOXYGEN_SHOULD_SKIP_THIS
//--------------------------------------------------------------------------
// Function:    H5Location::p_get_ref_obj_type (protected)
// Purpose      Retrieves the type of object that an object reference points to.
// Parameters
//              ref      - IN: Reference to query
//              ref_type - IN: Type of reference to query
// Return       An object type, which can be one of the following:
//                      H5O_TYPE_UNKNOWN        - Unknown object type (-1)
//                      H5O_TYPE_GROUP          - Object is a group
//                      H5O_TYPE_DATASET        - Object is a dataset
//                      H5O_TYPE_NAMED_DATATYPE - Object is a named datatype
//                      H5O_TYPE_NTYPES         - Number of object types
// Exception    H5::ReferenceException
// Programmer   Binh-Minh Ribler - May, 2004
//--------------------------------------------------------------------------
H5O_type_t
H5Location::p_get_ref_obj_type(void *ref, H5R_type_t ref_type) const
{
    H5O_type_t obj_type  = H5O_TYPE_UNKNOWN;
    herr_t     ret_value = H5Rget_obj_type2(getId(), ref_type, ref, &obj_type);
    if (ret_value < 0) {
        throw ReferenceException(inMemFunc("getRefObjType"), "H5Rget_obj_type2 failed");
    }
    if (obj_type == H5O_TYPE_UNKNOWN || obj_type >= H5O_TYPE_NTYPES) {
        throw ReferenceException(inMemFunc("getRefObjType"), "H5Rget_obj_type2 returned invalid type");
    }
    return (obj_type);
}

#endif // DOXYGEN_SHOULD_SKIP_THIS

//--------------------------------------------------------------------------
// Function:    H5Location::getRegion
///\brief       Retrieves a dataspace with the region pointed to selected.
///\param       ref      - IN: Reference to get region of
///\param       ref_type - IN: Type of reference to get region of - default
//                              to H5R_DATASET_REGION
///\return      DataSpace object
///\exception   H5::ReferenceException
// Programmer   Binh-Minh Ribler - May, 2004
// Modification
//      Mar 29, 2015
//              Used friend function to set id for DataSpace instead of the
//              existing id constructor or the setId method to avoid incrementing
//              ref count, as a work-around for a problem described in the JIRA
//              issue HDFFV-7947. -BMR
//--------------------------------------------------------------------------
DataSpace
H5Location::getRegion(void *ref, H5R_type_t ref_type) const
{
    hid_t space_id = H5Rget_region(getId(), ref_type, ref);
    if (space_id < 0) {
        throw ReferenceException(inMemFunc("getRegion"), "H5Rget_region failed");
    }
    try {
        DataSpace dataspace;
        f_DataSpace_setId(&dataspace, space_id);
        return (dataspace);
    }
    catch (DataSpaceIException &E) {
        throw ReferenceException(inMemFunc("getRegion"), E.getDetailMsg());
    }
}

//--------------------------------------------------------------------------
// Function:    H5Location::link
///\brief       Creates a soft link from \a link_name to \a target_name.
///\param       target_name - IN: Name of object, can be a non-existing object
///\param       link_name   - IN: Link name for the target name
///\param       lcpl - IN: Link creation plist - default to LinkCreatPropList::DEFAULT
///\param       lapl - IN: Link access plist - default to LinkAccPropList::DEFAULT
///\exception   One of the following:
///                 H5::GroupIException
///                 H5::FileIException
///                 H5::DataSetIException
///                 H5::DataTypeIException
///                 H5::AttributeIException
///\par Description
///             Note that both names are interpreted relative to the current
///             location.
///             For information on creating a soft link, please refer to the
///             H5Lcreate_soft APIs in the HDF5 C Reference Manual.
//  May 2018
//--------------------------------------------------------------------------
void
H5Location::link(const char *target_name, const char *link_name, const LinkCreatPropList &lcpl,
                 const LinkAccPropList &lapl) const
{
    herr_t ret_value = -1;
    hid_t  lcpl_id   = lcpl.getId();
    hid_t  lapl_id   = lapl.getId();

    ret_value = H5Lcreate_soft(target_name, getId(), link_name, lcpl_id, lapl_id);
    if (ret_value < 0)
        throwException("link", "creating soft link failed");
}

//--------------------------------------------------------------------------
// Function:    H5Location::link
///\brief       This is an overloaded member function, provided for convenience.
///             It differs from the above function in that it takes an
///             \c H5std_string for \a target_name and \a link_name.
// May 2018
//--------------------------------------------------------------------------
void
H5Location::link(const H5std_string &target_name, const H5std_string &link_name,
                 const LinkCreatPropList &lcpl, const LinkAccPropList &lapl) const
{
    link(target_name.c_str(), link_name.c_str(), lcpl, lapl);
}

//--------------------------------------------------------------------------
// Function:    H5Location::link
///\brief       Creates a hard link from \a new_name to \a curr_name.
///\param       curr_name - IN: Name of the existing object
///\param       new_loc   - IN: New location
///\param       new_name  - IN: New name for the object
///\param       lcpl - IN: Link creation plist - default to LinkCreatPropList::DEFAULT
///\param       lapl - IN: Link access plist - default to LinkAccPropList::DEFAULT
///\exception   One of the following:
///                 H5::GroupIException
///                 H5::FileIException
///                 H5::DataSetIException
///                 H5::DataTypeIException
///                 H5::AttributeIException
///\par Description
///             Note that both names are interpreted relative to the
///             specified location.
///             For information on creating a hard link, please refer to the
///             H5Lcreate_hard APIs in the HDF5 C Reference Manual.
//  May 2018
//--------------------------------------------------------------------------
void
H5Location::link(const char *curr_name, const H5Location &new_loc, const char *new_name,
                 const LinkCreatPropList &lcpl, const LinkAccPropList &lapl) const
{
    herr_t ret_value  = -1;
    hid_t  new_loc_id = new_loc.getId();
    hid_t  lcpl_id    = lcpl.getId();
    hid_t  lapl_id    = lapl.getId();

    ret_value = H5Lcreate_hard(getId(), curr_name, new_loc_id, new_name, lcpl_id, lapl_id);
    if (ret_value < 0)
        throwException("link", "creating link failed");
}

//--------------------------------------------------------------------------
// Function:    H5Location::link
///\brief       This is an overloaded member function, provided for convenience.
///             It differs from the above function in that it takes an
///             \c H5std_string for \a curr_name and \a new_name.
//  May 2018
//--------------------------------------------------------------------------
void
H5Location::link(const H5std_string &curr_name, const H5Location &new_loc, const H5std_string &new_name,
                 const LinkCreatPropList &lcpl, const LinkAccPropList &lapl) const
{
    link(curr_name.c_str(), new_loc, new_name.c_str(), lcpl, lapl);
}

//--------------------------------------------------------------------------
// Function:    H5Location::link
///\brief       Creates a hard link from \a new_name to \a curr_name - can be
///             used to pass in H5L_SAME_LOC.
///\param       curr_name - IN: Name of the existing object
///\param       same_loc  - IN: Location ID, or H5L_SAME_LOC
///\param       new_name  - IN: New name for the link
///\param       lcpl - IN: Link creation plist - default to LinkCreatPropList::DEFAULT
///\param       lapl - IN: Link access plist - default to LinkAccPropList::DEFAULT
///\exception   One of the following:
///                 H5::GroupIException
///                 H5::FileIException
///                 H5::DataSetIException
///                 H5::DataTypeIException
///                 H5::AttributeIException
///\par Description
///             Note that both names are interpreted relative to the
///             specified location.
///             For information on creating a hard link, please refer to the
///             H5Lcreate_hard APIs in the HDF5 C Reference Manual.
//  May 2018
//--------------------------------------------------------------------------
void
H5Location::link(const char *curr_name, const hid_t same_loc, const char *new_name,
                 const LinkCreatPropList &lcpl, const LinkAccPropList &lapl) const
{
    herr_t ret_value = -1;
    hid_t  lcpl_id   = lcpl.getId();
    hid_t  lapl_id   = lapl.getId();

    ret_value = H5Lcreate_hard(getId(), curr_name, same_loc, new_name, lcpl_id, lapl_id);

    if (ret_value < 0)
        throwException("link", "creating link failed");
}

//--------------------------------------------------------------------------
// Function:    H5Location::link
///\brief       This is an overloaded member function, provided for convenience.
///             It differs from the above function in that it takes an
///             \c H5std_string for \a curr_name and \a new_name.
//  May 2018
//--------------------------------------------------------------------------
void
H5Location::link(const H5std_string &curr_name, const hid_t same_loc, const H5std_string &new_name,
                 const LinkCreatPropList &lcpl, const LinkAccPropList &lapl) const
{
    link(curr_name.c_str(), same_loc, new_name.c_str(), lcpl, lapl);
}

//--------------------------------------------------------------------------
// Function:    H5Location::copyLink
///\brief       Copies a link from one location to another.
///\param       src_name - IN: Original name
///\param       dst      - IN: Destination location
///\param       dst_name - IN: New name
///\param       lcpl     - IN: Link creation plist - default LinkCreatPropList::DEFAULT
///\param       lapl     - IN: Link access plist - default LinkAccPropList::DEFAULT
///\exception   One of the following:
///                 H5::GroupIException
///                 H5::FileIException
///                 H5::DataSetIException
///                 H5::DataTypeIException
///                 H5::AttributeIException
//  May 2018
//--------------------------------------------------------------------------
void
H5Location::copyLink(const char *src_name, const H5Location &dst, const char *dst_name,
                     const LinkCreatPropList &lcpl, const LinkAccPropList &lapl) const
{
    herr_t ret_value;
    hid_t  dst_id  = dst.getId();
    hid_t  lcpl_id = lcpl.getId();
    hid_t  lapl_id = lapl.getId();

    ret_value = H5Lcopy(getId(), src_name, dst_id, dst_name, lcpl_id, lapl_id);
    if (ret_value < 0)
        throwException("copyLink", "H5Lcopy failed");
}

//--------------------------------------------------------------------------
// Function:    H5Location::copyLink
///\brief       This is an overloaded member function, provided for convenience.
///             It differs from the above function in that it takes an
///             \c H5std_string for \a src_name and \a dst_name.
//  May 2018
//--------------------------------------------------------------------------
void
H5Location::copyLink(const H5std_string &src_name, const H5Location &dst, const H5std_string &dst_name,
                     const LinkCreatPropList &lcpl, const LinkAccPropList &lapl) const
{
    copyLink(src_name.c_str(), dst, dst_name.c_str(), lcpl, lapl);
}

//--------------------------------------------------------------------------
// Function:    H5Location::copyLink
///\brief       Copies a link from a location into the same location.
///\param       src_name - IN: Original name
///\param       dst_name - IN: New name
///\param       lcpl     - IN: Link creation plist - default LinkCreatPropList::DEFAULT
///\param       lapl     - IN: Link access plist - default LinkAccPropList::DEFAULT
///\exception   One of the following:
///                 H5::GroupIException
///                 H5::FileIException
///                 H5::DataSetIException
///                 H5::DataTypeIException
///                 H5::AttributeIException
//  May 2018
//--------------------------------------------------------------------------
void
H5Location::copyLink(const char *src_name, const char *dst_name, const LinkCreatPropList &lcpl,
                     const LinkAccPropList &lapl) const
{
    herr_t ret_value;
    hid_t  lcpl_id = lcpl.getId();
    hid_t  lapl_id = lapl.getId();

    ret_value = H5Lcopy(getId(), src_name, H5L_SAME_LOC, dst_name, lcpl_id, lapl_id);
    if (ret_value < 0)
        throwException("copyLink", "H5Lcopy H5L_SAME_LOC failed");
}

//--------------------------------------------------------------------------
// Function:    H5Location::copyLink
///\brief       This is an overloaded member function, provided for convenience.
///             It differs from the above function in that it takes an
///             \c H5std_string for \a src_name and \a dst_name.
//  May 2018
//--------------------------------------------------------------------------
void
H5Location::copyLink(const H5std_string &src_name, const H5std_string &dst_name,
                     const LinkCreatPropList &lcpl, const LinkAccPropList &lapl) const
{
    copyLink(src_name.c_str(), dst_name.c_str(), lcpl, lapl);
}

//--------------------------------------------------------------------------
// Function:    H5Location::moveLink
///\brief       Renames a link in this location and moves it to a new location.
///\param       src_name - IN: Original name
///\param       dst      - IN: Destination location
///\param       dst_name - IN: New name
///\param       lcpl     - IN: Link creation plist - default LinkCreatPropList::DEFAULT
///\param       lapl     - IN: Link access plist - default LinkAccPropList::DEFAULT
///\exception   One of the following:
///                 H5::GroupIException
///                 H5::FileIException
///                 H5::DataSetIException
///                 H5::DataTypeIException
///                 H5::AttributeIException
///\note
///             Exercise care in moving groups as it is possible to render
///             data in a file inaccessible with H5Location::moveLink. Please refer
///             to the Group Interface in the HDF5 User's Guide for details.
//  May 2018
//--------------------------------------------------------------------------
void
H5Location::moveLink(const char *src_name, const H5Location &dst, const char *dst_name,
                     const LinkCreatPropList &lcpl, const LinkAccPropList &lapl) const
{
    herr_t ret_value;
    hid_t  dst_id  = dst.getId();
    hid_t  lcpl_id = lcpl.getId();
    hid_t  lapl_id = lapl.getId();

    ret_value = H5Lmove(getId(), src_name, dst_id, dst_name, lcpl_id, lapl_id);
    if (ret_value < 0)
        throwException("moveLink", "H5Lmove failed");
}

//--------------------------------------------------------------------------
// Function:    H5Location::moveLink
///\brief       This is an overloaded member function, provided for convenience.
///             It differs from the above function in that it takes an
///             \c H5std_string for \a src_name and \a dst_name.
//  May 2018
//--------------------------------------------------------------------------
void
H5Location::moveLink(const H5std_string &src_name, const H5Location &dst, const H5std_string &dst_name,
                     const LinkCreatPropList &lcpl, const LinkAccPropList &lapl) const
{
    moveLink(src_name.c_str(), dst, dst_name.c_str(), lcpl, lapl);
}

//--------------------------------------------------------------------------
// Function:    H5Location::moveLink
///\brief       Renames a link in this location.
///\param       src_name - IN: Original name
///\param       dst_name - IN: New name
///\param       lcpl     - IN: Link creation plist - default LinkCreatPropList::DEFAULT
///\param       lapl     - IN: Link access plist - default LinkAccPropList::DEFAULT
///\exception   One of the following:
///                 H5::GroupIException
///                 H5::FileIException
///                 H5::DataSetIException
///                 H5::DataTypeIException
///                 H5::AttributeIException
///\note
///             Exercise care in moving groups as it is possible to render
///             data in a file inaccessible with H5Location::moveLink. Please refer
///             to the Group Interface in the HDF5 User's Guide for details.
//  May 2018
//--------------------------------------------------------------------------
void
H5Location::moveLink(const char *src_name, const char *dst_name, const LinkCreatPropList &lcpl,
                     const LinkAccPropList &lapl) const
{
    herr_t ret_value;
    hid_t  lcpl_id = lcpl.getId();
    hid_t  lapl_id = lapl.getId();

    ret_value = H5Lmove(getId(), src_name, H5L_SAME_LOC, dst_name, lcpl_id, lapl_id);
    if (ret_value < 0)
        throwException("moveLink", "H5Lmove H5L_SAME_LOC failed");
}

//--------------------------------------------------------------------------
// Function:    H5Location::moveLink
///\brief       This is an overloaded member function, provided for convenience.
///             It differs from the above function in that it takes an
///             \c H5std_string for \a src_name and \a dst_name.
//  May 2018
//--------------------------------------------------------------------------
void
H5Location::moveLink(const H5std_string &src_name, const H5std_string &dst_name,
                     const LinkCreatPropList &lcpl, const LinkAccPropList &lapl) const
{
    moveLink(src_name.c_str(), dst_name.c_str(), lcpl, lapl);
}

//--------------------------------------------------------------------------
// Function:    H5Location::unlink
///\brief       Removes the specified link from this location.
///\param       name  - IN: Name of the object to be removed
///\param       lapl - IN: Link access plist - default to LinkAccPropList::DEFAULT
///\exception   One of the following:
///                 H5::GroupIException
///                 H5::FileIException
///                 H5::DataSetIException
///                 H5::DataTypeIException
///                 H5::AttributeIException
//  May 2018
// Developer's Note
//              Originally, CommonFG::unlink was the wrapper of H5Gunlink,
//              which was replaced by H5Ldelete.  The name "unlink" was kept
//              to help with backward compatibility.
//--------------------------------------------------------------------------
void
H5Location::unlink(const char *name, const LinkAccPropList &lapl) const
{
    herr_t ret_value = H5Ldelete(getId(), name, lapl.getId());
    if (ret_value < 0)
        throwException("unlink", "H5Ldelete failed");
}

//--------------------------------------------------------------------------
// Function:    H5Location::unlink
///\brief       This is an overloaded member function, provided for convenience.
///             It differs from the above function in that it takes an
///             \c H5std_string for \a name.
//  May 2018
//--------------------------------------------------------------------------
void
H5Location::unlink(const H5std_string &name, const LinkAccPropList &lapl) const
{
    unlink(name.c_str(), lapl);
}

//--------------------------------------------------------------------------
// Function:    H5Location::getLinkInfo
///\brief       Returns the information of the named link.
///\param       link_name  - IN: Symbolic link to the object
///\param       lapl       - IN: Link access plist - default LinkAccPropList::DEFAULT
///\return      Name of the object
///\exception   H5::FileIException/H5::GroupIException/H5::LocationException
// 2000
//--------------------------------------------------------------------------
H5L_info_t
H5Location::getLinkInfo(const char *link_name, const LinkAccPropList &lapl) const
{
    H5L_info_t linkinfo; // link info structure

    herr_t ret_value = H5Lget_info(getId(), link_name, &linkinfo, lapl.getId());
    if (ret_value < 0)
        throwException("getLinkInfo", "H5Lget_info to find buffer size failed");

    return (linkinfo);
}

//--------------------------------------------------------------------------
// Function:    H5Location::getLinkInfo
///\brief       This is an overloaded member function, provided for convenience.
///             It differs from the above function in that it takes an
///             \c H5std_string for \a link_name.
//--------------------------------------------------------------------------
H5L_info_t
H5Location::getLinkInfo(const H5std_string &link_name, const LinkAccPropList &lapl) const
{
    return (getLinkInfo(link_name.c_str(), lapl));
}

//--------------------------------------------------------------------------
// Function:    H5Location::getLinkval
///\brief       Returns the name of the object that the symbolic link points to.
///\param       name  - IN: Symbolic link to the object
///\param       size - IN: Maximum number of characters of value to be returned
///\return      Name of the object
///\exception   H5::FileIException/H5::GroupIException/H5::LocationException
// 2000
//--------------------------------------------------------------------------
H5std_string
H5Location::getLinkval(const char *name, size_t size) const
{
    H5L_info_t   linkinfo;
    char        *value_C; // value in C string
    size_t       val_size = size;
    H5std_string value    = "";
    herr_t       ret_value;

    // if user doesn't provide buffer size, determine it
    if (size == 0) {
        ret_value = H5Lget_info(getId(), name, &linkinfo, H5P_DEFAULT);
        if (ret_value < 0)
            throwException("getLinkval", "H5Lget_info to find buffer size failed");

        val_size = linkinfo.u.val_size;
    }

    // if link has value, retrieve the value, otherwise, return null string
    if (val_size > 0) {
        value_C = new char[val_size + 1];   // temporary C-string for C API
        HDmemset(value_C, 0, val_size + 1); // clear buffer

        ret_value = H5Lget_val(getId(), name, value_C, val_size, H5P_DEFAULT);
        if (ret_value < 0) {
            delete[] value_C;
            throwException("getLinkval", "H5Lget_val failed");
        }

        value = H5std_string(value_C);
        delete[] value_C;
    }
    return (value);
}

//--------------------------------------------------------------------------
// Function:    H5Location destructor
///\brief       Noop destructor.
// Programmer   Binh-Minh Ribler - 2000
//--------------------------------------------------------------------------
H5Location::~H5Location()
{
}

//--------------------------------------------------------------------------
// Function:    f_Attribute_setId - friend
// Modification:
//              Moved to H5Object.cpp after the rearrangement of classes
//              -BMR, Dec 2016
//--------------------------------------------------------------------------

//--------------------------------------------------------------------------
// Function:    f_DataSpace_setId - friend
// Purpose:     This function is friend to class H5::DataSpace so that it can
//              can set DataSpace::id in order to work around a problem
//              described in the JIRA issue HDFFV-7947.
//              Applications shouldn't need to use it.
// param        dspace   - IN/OUT: DataSpace object to be changed
// param        new_id - IN: New id to set
// Programmer   Binh-Minh Ribler - 2015
//--------------------------------------------------------------------------
void
f_DataSpace_setId(DataSpace *dspace, hid_t new_id)
{
    dspace->p_setId(new_id);
}

} // namespace H5
